package com.atid.app.atx.bluetooth.inventory.barcode.adapter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;

import com.atid.app.atx.bluetooth.inventory.barcode.R;
import com.atid.lib.module.barcode.types.BarcodeType;
import com.atid.lib.util.diagnotics.ATLog;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.LinearLayout;
import android.widget.TextView;

public class DataListAdapter extends BaseAdapter {
	private static final String TAG = DataListAdapter.class.getSimpleName();
	private static final int INFO = ATLog.L2;
	
	private LayoutInflater mInflater;
	private ArrayList<DataListItem> mList;
	private HashMap<String, DataListItem> mMap;
	
	public DataListAdapter(Context context) {
		super();
		
		mInflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		mList = new ArrayList<DataListItem>();
		mMap = new HashMap<String, DataListItem>();
		
	}
	
	public synchronized void clear() {
		mList.clear();
		mMap.clear();
		notifyDataSetChanged();
		
		ATLog.i(TAG, INFO, "INFO. clear()");
	}
	
	public synchronized void add(BarcodeType type, String codeId, String barcode) {
		DataListItem item = null;
		
		String key = String.format(Locale.US, "%s%s%s", type, codeId, barcode);
		if ((item = mMap.get(key)) == null) {
			item = new DataListItem(type, codeId, barcode);
			mMap.put(key, item);
			mList.add(item);
		} else {
			item.update();
		}
		notifyDataSetChanged();
	}
	
	public synchronized String getData(int position) {
		return mList.get(position).getData();
	}
	
	@Override
	public int getCount() {
		int size = 0;
		synchronized(mList) {
			size = mList.size();
		}
		return size;
	}
	
	@Override
	public Object getItem(int position) {
		DataListItem item = null;
		
		synchronized(this) {
			item = mList.get(position);
		}
		return item;
	}
	
	@Override
	public long getItemId(int position) {
		return position;
	}
	
	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		DataListViewHolder holder = null;
		DataListItem item = null;
		
		if (null == convertView) {
			convertView = mInflater.inflate(R.layout.item_data_list, parent, false);
			holder = new DataListViewHolder(convertView);
		} else {
			holder = (DataListViewHolder) convertView.getTag();
		}
		
		synchronized(this) {
			item = mList.get(position);
		}
		holder.display(item);
		return convertView;
	}

	// ------------------------------------------------------------------------
	// Declare Class DataListViewHolder
	// ------------------------------------------------------------------------
	private class DataListViewHolder {
		
		private LinearLayout linearBarcodeItem;
		private TextView txtBarcodeType;
		private TextView txtBarcodeId;
		private TextView txtData;
		private TextView txtCount;
		
		private DataListViewHolder(View parent) {
			linearBarcodeItem = (LinearLayout) parent.findViewById(R.id.linear_barcode_item);

			txtBarcodeType = (TextView) parent.findViewById(R.id.code_type);
			txtBarcodeId = (TextView) parent.findViewById(R.id.code_id);
			txtData = (TextView) parent.findViewById(R.id.data);
			
			txtCount = (TextView) parent.findViewById(R.id.count);
			parent.setTag(this);
		}
		
		public void display(DataListItem item) {
			if (null != item) {

				txtData.setText(item.getData());
				
				linearBarcodeItem.setVisibility(View.VISIBLE);
				txtBarcodeType.setText(item.getBarocdeType().toString());
				txtBarcodeId.setText(item.getBarcodeId());
				
				txtCount.setText(String.format(Locale.US, "%d", item.getCount()));
			}
		}
	}
	
	// ------------------------------------------------------------------------
	// Declare Class DataListItem
	// ------------------------------------------------------------------------
	private class DataListItem {
		private volatile String mData;
		private volatile BarcodeType mBarcodeType;
		private volatile String mBarcodeId;
		private volatile int mCount;
		
		private DataListItem(BarcodeType type, String codeId, String barcode) {
			mData = barcode;
			mBarcodeId = codeId;
			mBarcodeType = type;
			mCount = 1;
		}
		
		public String getData() {
			return mData;
		}
		
		public BarcodeType getBarocdeType() {
			return mBarcodeType;
		}
		
		public String getBarcodeId() {
			return mBarcodeId;
		}
		
		public int getCount() {
			return mCount;
		}
		
		public void update() {
			mCount++;
		}
		
		@Override
		public String toString() {
			return String.format(Locale.US, "[%s], [%s], [%s]", mData, mBarcodeType, mBarcodeId);
		}
	}
}
