package com.atid.app.atx.usb.inventory.key;

import java.util.ArrayList;
import java.util.Locale;

import com.atid.app.atx.usb.inventory.key.adapter.KeyListAdapter;
import com.atid.app.atx.usb.inventory.key.device.ReaderManager;
import com.atid.app.atx.usb.inventory.key.dialog.MessageBox;
import com.atid.app.atx.usb.inventory.key.dialog.MessageBoxCancel;
import com.atid.app.atx.usb.inventory.key.dialog.WaitDialog;
import com.atid.app.atx.usb.inventory.key.R;
import com.atid.lib.atx88.ATx88Reader;
import com.atid.lib.diagnostics.ATException;
import com.atid.lib.reader.ATEAReader;
import com.atid.lib.reader.event.IATEAReaderEventListener;
import com.atid.lib.reader.types.KeyState;
import com.atid.lib.reader.types.KeyType;
import com.atid.lib.reader.types.NotificationState;
import com.atid.lib.reader.types.OperationMode;
import com.atid.lib.reader.types.UsbChargerControl;
import com.atid.lib.reader.types.UsbChargerState;
import com.atid.lib.reader.types.UsbChargerType;
import com.atid.lib.transport.types.ConnectState;
import com.atid.lib.transport.types.ConnectType;
import com.atid.lib.types.ActionState;
import com.atid.lib.types.DeviceType;
import com.atid.lib.types.ResultCode;
import com.atid.lib.util.SysUtil;
import com.atid.lib.util.diagnotics.ATLog;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application.ActivityLifecycleCallbacks;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.ListView;
import android.widget.TextView;

public class MainActivity extends Activity implements IATEAReaderEventListener {
	private static final String TAG = MainActivity.class.getSimpleName();
	private static final int INFO = ATLog.L1;
	
	private static final int REQUEST_PERMISSION_CODE = 1000;
	
	// ------------------------------------------------------------------------
	// Member Variable
	// ------------------------------------------------------------------------
	private TextView txtVersion;
	private TextView txtDevice;
	private TextView txtAddress;

	private ListView lstKey;
	
	private TextView txtKeyType;
	private TextView txtKeyState;

	private KeyListAdapter adpKey;
	
	private ATEAReader mReader;
	
	private volatile boolean mIsInitialize;
	private volatile boolean mIsConnected;
	
	private boolean mIsRunAppHomeButton = true;
	private OperationMode mOperationMode;
	
	private PowerManager.WakeLock mWakeLock;
	private ActivityLifecycleManager mActivityLifecycleCallbacks;
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		mActivityLifecycleCallbacks = new ActivityLifecycleManager();
		getApplication().registerActivityLifecycleCallbacks(mActivityLifecycleCallbacks);
		
		super.onCreate(savedInstanceState);
		setContentView(R.layout.activity_main);
		
		SysUtil.setContext(this);		// must used the usb device
		
		checkPermission();
		
		ATLog.i(TAG, INFO, "INFO. onCreate()");
	}

	@SuppressLint("NewApi")
	private void checkPermission() {
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
			
			final String[] permissionList = new String[] {
					Manifest.permission.WAKE_LOCK
			};
			
			ArrayList<String> permissions = new ArrayList<String>();
			int grantResult = PackageManager.PERMISSION_GRANTED;
			for (int i = 0 ; i < permissionList.length ; i++) {
				grantResult = ContextCompat.checkSelfPermission(MainActivity.this, permissionList[i]);
				if (grantResult == PackageManager.PERMISSION_DENIED) {
					permissions.add(permissionList[i]);
				}
			}
			
			if (permissions.size() <= 0) {
				onPermissionResult(true);
			} else {
				String[] requestPermissions = new String[permissions.size()];
				//permissions.toArray(requestPermissions);
				ActivityCompat.requestPermissions(this, permissions.toArray(requestPermissions), REQUEST_PERMISSION_CODE);
				                                         
			}
		} else {
			onPermissionResult(true);
		}
	}	
	
	@Override
	protected void onDestroy() {
		
		closeActivity();
		
		if (mReader != null) {
			mReader.removeListener(this);
			mReader.destroy();
		}
		super.onDestroy();
		getApplication().unregisterActivityLifecycleCallbacks(mActivityLifecycleCallbacks);		
	}
	
	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		getMenuInflater().inflate(R.menu.main, menu);
		return true;
	}

	@Override 
	public boolean onPrepareOptionsMenu(Menu menu) {
		ATLog.i(TAG, INFO, "INFO. onPrepareOptionsMenu()");

		MenuItem menuAction = menu.findItem(R.id.menu_action);
		MenuItem menuAutoConnect = menu.findItem(R.id.menu_auto_connect);

		if (mReader != null) {
			
			if (mReader.getDeviceAttached()) {
				menuAction.setEnabled(true);
				menuAutoConnect.setEnabled(true);
				
				if (mReader.getState() == ConnectState.Connected) {
					menuAction.setTitle(R.string.action_disconnect);
				} else {
					menuAction.setTitle(R.string.action_connect);
				}
				
				if (mReader.getAutoConnect()) {
					menuAutoConnect.setTitle(R.string.action_disable_auto_connect);
				} else {
					menuAutoConnect.setTitle(R.string.action_enable_auto_connect);
				}

			} else {
				menuAction.setEnabled(false);
				menuAutoConnect.setEnabled(false);
			}
		} else {
			menuAction.setEnabled(false);
			menuAutoConnect.setEnabled(false);
		}
		
		return super.onPrepareOptionsMenu(menu);
	}
	
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {

		int id = item.getItemId();
		switch(id) {
		case R.id.menu_action:
			if (null != mReader) {
				if(mReader.getState() == ConnectState.Connected) {
					mReader.disconnect();
				} else {
					if (!mReader.connect() ) {
						ATLog.e(TAG, "ERROR. onOptionsItemSelected - Failed to connect reader");
						MessageBox.show( this, getString(R.string.msg_not_connect_usb),
								getString(R.string.title_error));
					}
				}				
			}
			return true;

		case R.id.menu_auto_connect:
			if (mReader.getAutoConnect()) {
				mReader.setAutoConnect(false);
			} else {
				mReader.setAutoConnect(true);
			}
			return true;
		default:
			break;
		}

		return super.onOptionsItemSelected(item);
	}
	
	@Override
	public void onUserLeaveHint() {
		
		if (!mIsRunAppHomeButton) {
			if (mReader != null) {
				closeActivity();
			}
		}
		ATLog.i(TAG, INFO, "INFO. onUserLeaveHint()");
		super.onUserLeaveHint();
	}
	
	@Override
	public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
		if (requestCode != REQUEST_PERMISSION_CODE) {
			return;
		}
		
		boolean result = true;
		for (int i = 0 ; i < grantResults.length ; i++) {
			result &= ( grantResults[i] != PackageManager.PERMISSION_DENIED );
		}
		onPermissionResult(result);
	}
	
	private void onPermissionResult(boolean result) {
		
		if (!result) {
			MessageBox.show(this, R.string.msg_fail_permission, R.string.title_error,
					android.R.drawable.ic_dialog_alert, new DialogInterface.OnClickListener() {
						
						@Override
						public void onClick(DialogInterface dialog, int which) {
							MainActivity.this.finish();
							return;
						}
					});
		}
		
		mReader = ReaderManager.getReader(ConnectType.USB, "", "", false);
		if (mReader == null) {
			MessageBox.show(this, R.string.msg_fail_load_reader_instance, 
					R.string.title_error, android.R.drawable.ic_dialog_alert,
					new DialogInterface.OnClickListener() {
				
				@Override
				public void onClick(DialogInterface dialog, int which) {
					MainActivity.this.finish();
					return;
				}
			});
			return;
		}
		// Set Reader Event Listener
		mReader.addListener(this);
		
		mIsInitialize = false;
		mIsConnected = false;
		
		initActivity();

		enableWidgets(false);
	}
	
	// ------------------------------------------------------------------------
	// Reader Event Handler Methods
	// ------------------------------------------------------------------------

	@Override
	public void onReaderStateChanged(ATEAReader reader, ConnectState state, Object params) {

		switch(state) {
		case Connected:
			
			if (mReader.getTransport().getConnectType() == ConnectType.USB) {
				txtDevice.setText(reader.getDeviceType().toString());
				txtAddress.setText(reader.getAddress());
			}
			
			WaitDialog.hide();
			mIsConnected = true;
			loadProperties();
			break;
		case Connecting:
			WaitDialog.show(this, R.string.msg_connecting_usb);
			break;			
		case Disconnected:
			WaitDialog.hide();
			mIsConnected = false;
			enableWidgets(false);
			closeActivity();
			break;
		default:
			break;
		}
		ATLog.i(TAG, INFO, "EVENT. onReaderStateChanged([%s], %s)", reader, state);
	}	
	
	@Override
	public void onReaderActionChanged(ATEAReader reader, ResultCode code, ActionState action, Object params) {
		
		ATLog.i(TAG, INFO, "EVENT. onReaderActionChanged([%s], %s, %s)", reader, code, action);
	}

	@Override
	public void onReaderOperationModeChanged(ATEAReader reader, OperationMode mode, Object params) {
		
		mOperationMode = mode;
		
		switch(mode) {
		case KeyEventOnly:
			enableWidgets(true);
			break;
		case Normal:
		case Barcode:
		default:
			enableWidgets(false);
			break;
		}
		ATLog.i(TAG, INFO, "EVENT. onReaderOperationModeChanged([%s], %s)", reader, mode);
	}	

	@Override
	public void onReaderBatteryState(ATEAReader reader, int batteryState, Object params) {
		
		ATLog.i(TAG, INFO, "EVENT. onReaderBatteryState([%s], %d)", reader, batteryState);
	}

	@Override
	public void onReaderKeyChanged(ATEAReader reader, KeyType type, KeyState state, Object params) {
		
		switch(mOperationMode) {
		case KeyEventOnly:
			enableWidgets(true);
			break;
		case Normal:
		case Barcode:
		default:
			enableWidgets(false);
			break;
		}

		adpKey.add(type.toString(), state.toString());
		txtKeyType.setText(type.toString());
		txtKeyState.setText(state.toString());
		
		ATLog.i(TAG, INFO, "EVENT. onReaderKeyChanged([%s], %s, %s)", reader, type, state);		
	}
	
	@Override
	public void onReaderUsbChargerChanged(ATEAReader reader, UsbChargerType type, UsbChargerState state, Object params) {

		if (reader.getTransport().getConnectType() == ConnectType.USB) {

			final ATx88Reader x88Reader = (ATx88Reader) reader;
			if (state == UsbChargerState.Insert) {
				
				MessageBoxCancel.show(this, R.string.msg_usb_charigng_on, R.string.title_usb_charging,
						android.R.drawable.ic_menu_info_details, new DialogInterface.OnClickListener() {
							
							@Override
							public void onClick(DialogInterface dialog, int which) {

								try {
									x88Reader.setUsbChargingPhone(UsbChargerControl.Charging);
								} catch (ATException e) {
									ATLog.e(TAG, e, "ERROR. $onReaderUsbChargerChanged().onClick - %s",
											R.string.msg_fail_set_usb_charging);
									MessageBoxCancel.hide();
									showMessage(String.format(Locale.US, "%s\r\nError[%s]",
											getResources().getString(R.string.msg_fail_set_usb_charging), e.getCode()));
									
									return;
								}
								
								MessageBoxCancel.hide();
							}
						}, new DialogInterface.OnCancelListener() {
							
							@Override
							public void onCancel(DialogInterface dialog) {
								try {
									x88Reader.setUsbChargingPhone(UsbChargerControl.NoCharging);
								} catch (ATException e) {
									ATLog.e(TAG, e, "ERROR. $onReaderUsbChargerChanged().onCancel() - %s",
											R.string.msg_fail_set_usb_charging);
									MessageBoxCancel.hide();
									showMessage(String.format(Locale.US, "%s\r\nError[%s]",
											getResources().getString(R.string.msg_fail_set_usb_charging), e.getCode()));
									
									return;
								}

								MessageBoxCancel.hide();
							}
						});

			} else {
				MessageBoxCancel.hide();
			}				
		}
		
		ATLog.i(TAG, INFO, "EVENT. onReaderUsbChargerChanged([%s], %s, %s)", reader, type, state);
	}
	
	// ------------------------------------------------------------------------
	// Internal Widget Control Methods
	// ------------------------------------------------------------------------
	private void initActivity() {
		txtVersion = (TextView) findViewById(R.id.app_version);
		txtVersion.setText(SysUtil.getVersion(this));
		txtDevice = (TextView) findViewById(R.id.device);
		txtAddress = (TextView) findViewById(R.id.address);
		
		lstKey = (ListView) findViewById(R.id.key_list);
		adpKey = new KeyListAdapter(this);
		lstKey.setAdapter(adpKey);
		
		txtKeyType = (TextView) findViewById(R.id.key_type);
		txtKeyState = (TextView) findViewById(R.id.key_state);

	}
	
	private void closeActivity() {
		WaitDialog.hide();
		
		ATLog.i(TAG, INFO, "INFO. closeActivity()");
	}
	
	private void enableWidgets(boolean enabled) {
		
		lstKey.setEnabled(enabled);
		
		txtKeyType.setEnabled(enabled);
		txtKeyState.setEnabled(enabled);
		
		txtDevice.setEnabled(enabled || mIsConnected);
		txtAddress.setEnabled(enabled || mIsConnected);
		
		ATLog.i(TAG, INFO, "INFO. enableWidgets(%s)", enabled);
	}
	
	private void showMessage(final String message) {
		runOnUiThread( new Runnable() {
			@Override
			public void run() {
				MessageBox.show(MainActivity.this, message);
			}
		});
		
	}
	
	// ------------------------------------------------------------------------
	// Load RFID Reader Properties
	// ------------------------------------------------------------------------
	private void loadProperties() {
		WaitDialog.show(this, R.string.msg_initialize_view);
		
		Thread thread = new Thread(mLoadingProperties);
		thread.start();
	}
	
	private Runnable mLoadingProperties = new Runnable() {
		@Override
		public void run() {
			mIsInitialize = loadingProperties();
			runOnUiThread(mLoadedProperties);
		}
	};
	
	private Runnable mLoadedProperties = new Runnable() {
		@Override
		public void run() {
			loadedProperties(mIsInitialize);
			WaitDialog.hide();
		}
	};	
	
	private boolean loadingProperties() {
		
		if (mIsConnected) {
			if (mReader.getDeviceType() != DeviceType.ATS100) {
				
				runOnUiThread(new Runnable(){
					@Override
					public void run() {
						MessageBox.show(MainActivity.this, getString(R.string.msg_not_support_function),
								getString(R.string.title_error));
					}
				});
				ATLog.e(TAG, "ERROR. loadingProperties() - %s", getString(R.string.msg_not_support_function));
				return false;
			}

		} else {
			ATLog.e(TAG, "ERROR. loadingProperties() - disconnected the usb ");
			return false;
		}
		
		OperationMode mode = OperationMode.Normal;
		
		if (mIsConnected) {
			try {
				mode = mReader.getOperationMode();
			} catch (ATException e) {
				ATLog.e(TAG, e, "ERROR. loadingProperties() - Failed to load operation mode");
				return false;
			}
		} else {
			ATLog.e(TAG, "ERROR. loadingProperties() - disconnected the usb ");
			return false;
		}
		
		if (mIsConnected) {
			switch(mode) {
			case KeyEventOnly:
				break;
			case Barcode:
			case Normal:
				try {
					mReader.setOperationMode(OperationMode.KeyEventOnly);
				} catch (ATException e) {
					ATLog.e(TAG, e, "ERROR. loadingProperties() - Failed to set operation mode");
					return false;
				}
				break;
			default:
				ATLog.e(TAG, "ERROR. loadingProperties() - Unknown operation mode");
				return false;
			}
		} else {
			ATLog.e(TAG, "ERROR. loadingProperties() - disconnected the usb ");
			return false;
		}
		
		mOperationMode = OperationMode.KeyEventOnly;
		
		// Enabled Use Key Action
		if (mIsConnected) {
			try {
				mReader.setUseActionKey(true);
			} catch (ATException e) {
				ATLog.e(TAG, "ERROR. loadingProperties() - Failed to enabled key action");
				return false;
			}
		} else {
			ATLog.e(TAG, "ERROR. loadingProperties() - disconnected the usb ");
			return false;
		}
		
		ATLog.i(TAG, INFO, "INFO. loadingProperties()");
		return true;
	}
	
	private void loadedProperties(boolean isInitialize) {
		
		if (isInitialize) {
			enableWidgets(true);
		} else {
			enableWidgets(false);
		}
		
		ATLog.i(TAG, INFO, "INFO. loadedProperties(%s)", isInitialize);
	}	
	
	// ------------------------------------------------------------------------
	// Activity Life cycle
	// ------------------------------------------------------------------------
	public class ActivityLifecycleManager implements ActivityLifecycleCallbacks {

		private int mRefCount = 0;
		private String _tag = ActivityLifecycleManager.class.getSimpleName();
		
		@SuppressWarnings("deprecation")
		@Override
		public void onActivityStarted(Activity activity) {
			mRefCount++;
			ATLog.i(TAG, INFO, "INFO. ActivityLifecycleManager.onActivityStarted : " + mRefCount);
			
			if(mRefCount == 1) {
				// Setup always wake up
				android.content.Context context = activity.getApplicationContext();
				//SysUtil.wakeLock(activity.getApplicationContext(), SysUtil.getAppName(context));
				
				if(mWakeLock != null) {
					ATLog.i(TAG, INFO, "INFO. Already exist wake lock");
				}
				
				PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
				mWakeLock = powerManager.newWakeLock(PowerManager.FULL_WAKE_LOCK, _tag);
				//mWakeLock = powerManager.newWakeLock(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON, TAG);
				mWakeLock.acquire();
				ATLog.i(TAG, INFO, "INFO. Acquires the wake lock.");
			}
		}
		
		@Override
		public void onActivityStopped(Activity activity) {
			mRefCount--;
			ATLog.i(TAG, INFO, "INFO. ActivityLifecycleManager.onActivityStopped : " + mRefCount);
			
			if(mRefCount == 0) {
				// release WakeLock.
				//SysUtil.wakeUnlock();
				
				if (mWakeLock == null)
					return;

				mWakeLock.release();
				mWakeLock = null;
				ATLog.i(TAG, INFO, "INFO. Releases the wake lock.");
			}
		}

		@Override
		public void onActivityCreated(Activity activity, Bundle savedInstanceState) {}
		@Override
		public void onActivityResumed(Activity activity) {}
		@Override
		public void onActivityPaused(Activity activity) {}
		@Override
		public void onActivitySaveInstanceState(Activity activity, Bundle outState) {}
		@Override
		public void onActivityDestroyed(Activity activity) {}
		
	}		
}
