package com.atid.app.atx.usb.inventory.rfid.adapter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;

import com.atid.app.atx.usb.inventory.rfid.R;
import com.atid.lib.util.diagnotics.ATLog;

import android.content.Context;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.BaseAdapter;
import android.widget.LinearLayout;
import android.widget.TextView;

public class DataListAdapter extends BaseAdapter {
	private static final String TAG = DataListAdapter.class.getSimpleName();
	private static final int INFO = ATLog.L2;
	
	private LayoutInflater mInflater;
	private ArrayList<DataListItem> mList;
	private HashMap<String, DataListItem> mMap;
	
	private volatile boolean mIsDisplayPC;
	private volatile boolean mIsReportRssi;
	
	public DataListAdapter(Context context) {
		super();
		
		mInflater = (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		mList = new ArrayList<DataListItem>();
		mMap = new HashMap<String, DataListItem>();
		
		mIsDisplayPC = true;
		mIsReportRssi = false;
	}
	
	public synchronized void clear() {
		mList.clear();
		mMap.clear();
		notifyDataSetChanged();
		
		ATLog.i(TAG, INFO, "INFO. clear()");
	}
	
	public synchronized void add(String tag, float rssi, float phase) {
		
		DataListItem item = null;
		
		if ( (item = mMap.get(tag)) == null ) {
			item = new DataListItem(tag, rssi, phase);
			mMap.put(tag, item);
			mList.add(item);
		} else {
			item.update(rssi, phase);
		}
		notifyDataSetChanged();
	}
	
	public void setDisplayPC(boolean enabled) {
		mIsDisplayPC = enabled;
		notifyDataSetChanged();		
	}
	
	public boolean getDisplayPC() {
		return mIsDisplayPC;
	}
	
	public void setReportRssi(boolean enabled) {
		mIsReportRssi = enabled;
		notifyDataSetChanged();		
	}
	
	public boolean getReportRssi() {
		return mIsReportRssi;
	}
	
	public synchronized String getData(int position) {
		return mList.get(position).getData();
	}
	
	@Override
	public int getCount() {
		int size = 0;
		synchronized(mList) {
			size = mList.size();
		}
		return size;
	}
	
	@Override
	public Object getItem(int position) {
		DataListItem item = null;
		
		synchronized(this) {
			item = mList.get(position);
		}
		return item;
	}
	
	@Override
	public long getItemId(int position) {
		return position;
	}
	
	@Override
	public View getView(int position, View convertView, ViewGroup parent) {
		DataListViewHolder holder = null;
		DataListItem item = null;
		
		if (null == convertView) {
			convertView = mInflater.inflate(R.layout.item_data_list, parent, false);
			holder = new DataListViewHolder(convertView);
		} else {
			holder = (DataListViewHolder) convertView.getTag();
		}
		
		synchronized(this) {
			item = mList.get(position);
		}
		holder.display(item);
		return convertView;
	}

	// ------------------------------------------------------------------------
	// Declare Class DataListViewHolder
	// ------------------------------------------------------------------------
	private class DataListViewHolder {
		
		private LinearLayout linearTagItem;
		private TextView txtData;
		private TextView txtRssi;
		private TextView txtPhase;
		private TextView txtCount;
		
		private DataListViewHolder(View parent) {
			linearTagItem = (LinearLayout) parent.findViewById(R.id.linear_tag_item);
			
			txtData = (TextView) parent.findViewById(R.id.data);
			txtRssi = (TextView) parent.findViewById(R.id.rssi);
			txtPhase = (TextView) parent.findViewById(R.id.phase);
			txtCount = (TextView) parent.findViewById(R.id.count);
			parent.setTag(this);
		}
		
		public void display(DataListItem item) {
			if (null != item) {
				txtData.setText(mIsDisplayPC ? item.getData() :
					item.getData().length() > 4 ? item.getData().substring(4) : item.getData());
				
				if (mIsReportRssi) {
					linearTagItem.setVisibility(View.VISIBLE);
					
					txtRssi.setText(String.format(Locale.US, "%.2f dB", item.getRssi()));
					txtPhase.setText(String.format(Locale.US, "%.2f \u00B0", item.getPhase()));
				} else {
					linearTagItem.setVisibility(View.GONE);
				}
				
				txtCount.setText(String.format(Locale.US, "%d", item.getCount()));
			}
		}
	}
	
	// ------------------------------------------------------------------------
	// Declare Class DataListItem
	// ------------------------------------------------------------------------
	private class DataListItem {
		private volatile String mData;
		private volatile float mRssi;
		private volatile float mPhase;
		private volatile int mCount;
		
		private DataListItem(String tag, float rssi, float phase) {
			mData = tag;
			mRssi = rssi;
			mPhase = phase;
			mCount = 1;
		}
		
		public String getData() {
			return mData;
		}
		
		public float getRssi() {
			return mRssi;
		}
		
		public float getPhase() {
			return mPhase;
		}
		
		public int getCount() {
			return mCount;
		}
		
		@SuppressWarnings("unused")
		public void update() {
			mCount++;
		}
		
		public void update(float rssi, float phase) {
			mRssi = rssi;
			mPhase = phase;
			mCount++;
		}
		
		@Override
		public String toString() {
			return String.format(Locale.US, "[%s], %.2f, %.2f", mData, mRssi, mPhase);
		}
	}
}
